<?php
/**
 * XFileSharingPro 2.0+ => Yetishare 5.4+ Migration Script
 *
 * Migration script for converting users, files and stats data from XFileSharingPro 2.0+. Set the config values below, upload this
 * script to the base of your Yetishare install and load it within a browser
 *
 * REQUIREMENTS:
 * MySQL PDO
 * Yetishare installed
 *
 * This has been tested with XFileSharingPro 2.0, although it may also work with other versions
 */

// XFileSharingPro 2.0 - database settings
const XFILESHARING_DB_HOST = 'localhost';
const XFILESHARING_DB_NAME = '';
const XFILESHARING_DB_USER = '';
const XFILESHARING_DB_PASS = '';

// XFileSharingPro password hash. Needed to decode user passwords
// for migration. This value is in XFileConfig.pm in password_salt.
// i.e. password_salt  => '',
const XFILESHARING_PASSWORD_HASH = '[hash]';

// XFileSharingPro grouping for file storage. This value is in XFileConfig.pm in files_per_folder. Normally 5000. Set to 0 to avoid this migration appending the folder names to the original file. i.e. 00001, 00002, 00003 etc.
const XFILESHARING_FOLDER_GROUPING = '5000';

/**
 * ******************************************************************
 * END OF CONFIG SECTION, YOU SHOULDN'T NEED TO CHANGE ANYTHING ELSE
 * ******************************************************************
 */

// other settings
const SOURCE_APP_NAME = 'XFileSharingPro';
const ENABLED_SECTIONS = 'servers,files,users,folders,payments'; // i.e. servers,files,users,folders,payments
const SOURCE_TABLE_NAME_SERVERS = 'Servers';
const SOURCE_TABLE_NAME_FILES = 'Files';
const SOURCE_TABLE_NAME_USERS = 'Users';
const SOURCE_TABLE_NAME_FOLDERS = 'Folders';
const SOURCE_TABLE_NAME_PAYMENTS = 'Payments';

// Yetishare - config file path
const YETISHARE_CONFIG_FILE_PATH = '_config.inc.php';

// allow up to 24 hours for it to run
set_time_limit(60 * 60 * 24);

// make sure we are in the root and can find the config file
if (!file_exists(YETISHARE_CONFIG_FILE_PATH)) {
    die('ERROR: Could not load Yetishare config file. Ensure you\'re running this script from the root of your Yetishare install.');
}

// include Yetishare config
require_once(YETISHARE_CONFIG_FILE_PATH);

// test database connectivity, YetiShare
try {
    $ysDBH = new PDO('mysql:host='._CONFIG_DB_HOST.';dbname='._CONFIG_DB_NAME, _CONFIG_DB_USER, _CONFIG_DB_PASS);
    $ysDBH->exec('set names utf8');
} catch (PDOException $e) {
    die('ERROR: Could not connect to Yetishare database. '.$e->getMessage());
}

// test database connectivity, source
try {
    $sourceDBH = new PDO('mysql:host='.XFILESHARING_DB_HOST.';dbname='.XFILESHARING_DB_NAME, XFILESHARING_DB_USER,
        XFILESHARING_DB_PASS);
    $sourceDBH->exec('set names utf8');
} catch (PDOException $e) {
    die('ERROR: Could not connect to '.SOURCE_APP_NAME.' database. '.$e->getMessage());
}

// initial checks passed, load stats for converting and get user confirmation
$sourceDBStats = [];

// servers
if (isSectionEnabled('servers')) {
    $getFiles = $sourceDBH->query('SELECT count(*) AS total FROM '.SOURCE_TABLE_NAME_SERVERS);
    $row = $getFiles->fetchObject();
    $sourceDBStats['totalServers'] = (int)$row->total;
}

// files
if (isSectionEnabled('files')) {
    $getFiles = $sourceDBH->query('SELECT count(*) AS total FROM '.SOURCE_TABLE_NAME_FILES);
    $row = $getFiles->fetchObject();
    $sourceDBStats['totalFiles'] = (int)$row->total;
}

// users
if (isSectionEnabled('users')) {
    $getUsers = $sourceDBH->query('SELECT count(*) AS total FROM '.SOURCE_TABLE_NAME_USERS);
    $row = $getUsers->fetchObject();
    $sourceDBStats['totalUsers'] = (int)$row->total;
}

// folders
if (isSectionEnabled('folders')) {
    $getFolders = $sourceDBH->query('SELECT count(*) AS total FROM '.SOURCE_TABLE_NAME_FOLDERS);
    $row = $getFolders->fetchObject();
    $sourceDBStats['totalFolders'] = (int)$row->total;
}

// payments
if (isSectionEnabled('payments')) {
    $getPayments = $sourceDBH->query('SELECT count(*) AS total FROM '.SOURCE_TABLE_NAME_PAYMENTS);
    $row = $getPayments->fetchObject();
    $sourceDBStats['totalPayments'] = (int)$row->total;
}

// page setup
const PAGE_TITLE = SOURCE_APP_NAME.' => Yetishare 5.4+ Migration Tool';
?>
    <html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">
    <head>
        <meta http-equiv="content-type" content="text/html; charset=utf-8"/>
        <title><?php echo PAGE_TITLE; ?></title>
        <meta name="distribution" content="global"/>
        <style>
            blockquote, code, table th {
                background: #ecf7fd
            }

            #navigation ul, body {
                margin: 0;
                padding: 0
            }

            #footer, #header, body {
                background: #ccc
            }

            #content, #footer, #header {
                padding: 20px
            }

            .button, .success, table caption {
                font-weight: 700
            }

            body {
                font: 100%/1.0 helvetica, arial, sans-serif;
                color: #444
            }

            h1, h2, h3, h4, h5, h6 {
                margin: 0 0 1em;
                line-height: 1.1
            }

            h2, h3 {
                color: #003d5d
            }

            h2 {
                font-size: 218.75%
            }

            h3 {
                font-size: 137.5%
            }

            h4 {
                font-size: 118.75%
            }

            h5 {
                font-size: 112.5%
            }

            p {
                margin: 0 0 1em
            }

            img {
                border: none
            }

            a:link {
                color: #035389
            }

            a:visited {
                color: #09619c
            }

            a:focus {
                color: #fff;
                background: #000
            }

            a:hover {
                color: #000
            }

            a:active {
                color: #c00;
                background: #fff
            }

            table {
                margin: 1em 0;
                border-collapse: collapse;
                width: 100%
            }

            table caption {
                text-align: left;
                padding: 0 0 5px;
                text-transform: uppercase;
                color: #236271
            }

            table td, table th {
                text-align: left;
                border: 1px solid #b1d2e4;
                padding: 5px 10px;
                vertical-align: top
            }

            blockquote {
                margin: 1em 0;
                padding: 1.5em
            }

            code {
                font: 115% courier, monaco, monospace;
                margin: 0 .3em
            }

            abbr, acronym {
                border-bottom: .1em dotted;
                cursor: help
            }

            #container {
                margin: 0;
                background: #fff
            }

            #header h1 {
                margin: 0
            }

            #navigation {
                float: left;
                width: 100%;
                background: #333
            }

            #navigation ul li {
                list-style-type: none;
                display: inline
            }

            #navigation li a {
                display: block;
                float: left;
                padding: 5px 10px;
                color: #fff;
                text-decoration: none;
                border-right: 1px solid #fff
            }

            #navigation li a:hover {
                background: #383
            }

            #content {
                clear: left
            }

            #content h2 {
                color: #000;
                font-size: 160%;
                margin: 0 0 .5em
            }

            #footer {
                text-align: right;
                height: 1%;
                font-size: 12px
            }

            .error, .important {
                color: red;
                font-weight: 700
            }

            .success {
                color: green
            }

            .button {
                border: 1px solid #4b546a;
                -webkit-box-shadow: #b7b8b8 0 1px 0 inset;
                -moz-box-shadow: #b7b8b8 0 1px 0 inset;
                box-shadow: #b7b8b8 0 1px 0 inset;
                padding: 10px;
                text-decoration: none;
                display: inline-block;
                text-shadow: -1px -1px 0 rgba(0, 0, 0, .3);
                color: #fff;
                background-color: #606c88;
                background-image: -webkit-gradient(linear, left top, left bottom, from(#606c88), to(#3f4c6b));
                background-image: -webkit-linear-gradient(top, #606c88, #3f4c6b);
                background-image: -moz-linear-gradient(top, #606c88, #3f4c6b);
                background-image: -ms-linear-gradient(top, #606c88, #3f4c6b);
                background-image: -o-linear-gradient(top, #606c88, #3f4c6b);
                background-image: linear-gradient(to bottom, #606c88, #3f4c6b);
                cursor: pointer
            }

            .button:hover {
                border: 1px solid #4b546a;
                background-color: #4b546a;
                background-image: -webkit-gradient(linear, left top, left bottom, from(#4b546a), to(#2c354b));
                background-image: -webkit-linear-gradient(top, #4b546a, #2c354b);
                background-image: -moz-linear-gradient(top, #4b546a, #2c354b);
                background-image: -ms-linear-gradient(top, #4b546a, #2c354b);
                background-image: -o-linear-gradient(top, #4b546a, #2c354b);
                background-image: linear-gradient(to bottom, #4b546a, #2c354b)
            }
        </style>
    </head>
    <body>
    <div id="container">
        <div id="header">
            <h1>
                <?php echo PAGE_TITLE; ?>
            </h1>
        </div>
        <div id="content">
            <?php
            if (!isset($_REQUEST['submitted'])): ?>
                <h2>
                    Confirm Migration
                </h2>
                <p>
                    Use this tool to migrate your user, files, folders and payments data
                    from <?php echo SOURCE_APP_NAME; ?> into a Yetishare install.
                </p>
                <p>
                    To start, upload this file to the root of your Yetishare install, ensure you've set your
                    configuration at the top of this php script, then click 'start migration' below. To confirm, we've
                    loaded your existing <?php echo SOURCE_APP_NAME; ?> table sizes below.
                </p>
                <p style='padding-top: 4px; padding-bottom: 4px;'>
                <table style='width: auto;'>
                    <tr>
                        <th style='width: 150px;'>Src Table:</th>
                        <th style='width: 150px;'>Total Rows:</th>
                    </tr>
                    <?php
                    if (isSectionEnabled('servers')): ?>
                        <tr>
                            <td>Servers:</td>
                            <td><?php echo $sourceDBStats['totalServers']; ?></td>
                        </tr>
                    <?php
                    endif; ?>
                    <?php
                    if (isSectionEnabled('files')): ?>
                        <tr>
                            <td>Files:</td>
                            <td><?php echo $sourceDBStats['totalFiles']; ?></td>
                        </tr>
                    <?php
                    endif; ?>
                    <?php
                    if (isSectionEnabled('users')): ?>
                        <tr>
                            <td>Users:</td>
                            <td><?php echo $sourceDBStats['totalUsers']; ?></td>
                        </tr>
                    <?php
                    endif; ?>
                    <?php
                    if (isSectionEnabled('folders')): ?>
                        <tr>
                            <td>Folders:</td>
                            <td><?php echo $sourceDBStats['totalFolders']; ?></td>
                        </tr>
                    <?php
                    endif; ?>
                    <?php
                    if (isSectionEnabled('payments')): ?>
                        <tr>
                            <td>Payments: (paid)</td>
                            <td><?php echo $sourceDBStats['totalPayments']; ?></td>
                        </tr>
                    <?php
                    endif; ?>
                </table>
                </p>
                <p class="important">
                    IMPORTANT: When you start this process, any existing data in your Yetishare database will be
                    cleared. Please ensure you've backed up both databases beforehand, so you can easily revert if you
                    need to.
                </p>
                <p class="important">
                    This process wont actually migrate your files, it converts all the data in
                    your <?php echo SOURCE_APP_NAME; ?>>
                    database for Yetishare. Although it does keep the same file names for your stored files. So after
                    this is completed you should move all your files into the Yetishare /files/ folder on each server.
                </p>
                <p style="padding-top: 4px;">
                <form method="POST" action="migrate.php">
                    <input type="hidden" name="submitted" value="1"/>
                    <input type="submit" value="Start Migration" name="submit" class="button"
                           onClick="return confirm('Are you sure you want to delete all the data from your Yetishare database and import from the <?php echo SOURCE_APP_NAME; ?> database?');"/>
                </form>
                </p>
            <?php else: ?>
                <h2>
                    Importing Data
                </h2>
                <p>
                    Clearing existing Yetishare data...
                    <?php
                    // delete Yetishare data
                    $ysDBH->query('DELETE FROM download_tracker');
                    $ysDBH->query('DELETE FROM file');
                    $ysDBH->query('DELETE FROM file_artifact');
                    $ysDBH->query('DELETE FROM file_artifact_storage');
                    $ysDBH->query('DELETE FROM file_folder');
                    $ysDBH->query('DELETE FROM payment_log');
                    $ysDBH->query('DELETE FROM sessions');
                    $ysDBH->query('DELETE FROM stats');
                    $ysDBH->query('DELETE FROM users');
                    $ysDBH->query('DELETE FROM file_server');
                    echo 'done.';

                    updateScreen();
                    ?>
                </p>
                <p style='padding-top: 4px; padding-bottom: 4px;'>
                <table style='width: auto;'>
                    <tr>
                        <th style='width: 150px;'>Src Table:</th>
                        <th style='width: 150px;'>Total Rows:</th>
                        <th style='width: 150px;'>Yetishare Table:</th>
                        <th style='width: 150px;'>Successful Rows:</th>
                        <th style='width: 150px;'>Failed Rows:</th>
                    </tr>

                    <?php
                    // do servers
                    if (isSectionEnabled('servers')) {
                        $sql = sprintf('SELECT srv_id as server_id,
                                   srv_name       as server_name,
                                   srv_ip         as server_ip,
                                   srv_htdocs_url as domain_name,
                                   srv_status     as status,
                                   srv_disk       as disk_usage,
                                   srv_disk_max   as max_storage
                            FROM %s',
                            SOURCE_TABLE_NAME_SERVERS);
                        $getServers = $sourceDBH->query($sql);

                        $success = 0;
                        $error = 0;
                        while ($row = $getServers->fetch()) {
                            $statusId = 2;
                            if ($row['status'] == 'READONLY') {
                                $statusId = 3;
                            }

                            $domain = $row['domain_name'];
                            $domainArr = parse_url($domain);
                            $domain = $domainArr['host'];

                            // insert into Yetishare db
                            $sql = 'INSERT INTO file_server (id, serverLabel, serverType, ipAddress, statusId, storagePath, fileServerDomainName,
                                                     scriptPath, totalSpaceUsed, maximumStorageBytes, ftpUsername)
                            VALUES (:id, :serverLabel, \'local\', :ipAddress, :statusId, \'files/\', :fileServerDomainName, \'/\', :totalSpaceUsed,
                                    :maximumStorageBytes, "")';
                            $q = $ysDBH->prepare($sql);
                            $count = $q->execute([
                                ':id' => $row['server_id'],
                                ':serverLabel' => $row['server_name'],
                                ':ipAddress' => $row['server_ip'],
                                ':statusId' => $statusId,
                                ':fileServerDomainName' => $domain,
                                ':totalSpaceUsed' => $row['disk_usage'],
                                ':maximumStorageBytes' => $row['max_storage'],
                            ]);

                            if ($count) {
                                $success++;
                            } else {
                                $error++;
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo SOURCE_TABLE_NAME_SERVERS; ?>:</td>
                            <td><?php echo $sourceDBStats['totalServers']; ?></td>
                            <td>file_server:</td>
                            <td><?php echo $success; ?></td>
                            <td><?php echo $error; ?></td>
                        </tr>
                        <?php
                        updateScreen();
                    }
                    ?>

                    <?php
                    // do files
                    if (isSectionEnabled('files')) {
                        $sql = sprintf('SELECT 
                                   file_id            as file_id,
                                   usr_id             as user_id,
                                   file_name          as file_name,
                                   file_code          as short_url,
                                   file_fld_id        as folder_id,
                                   file_real          as file_path,
                                   file_del_id        as delete_hash,
                                   file_downloads     as total_downloads,
                                   file_size          as file_size,
                                   file_password      as file_password,
                                   file_ip            as uploaded_ip,
                                   file_last_download as last_downloaded,
                                   file_created       as created,
                                   srv_id             as server_id
                            FROM %s',
                            SOURCE_TABLE_NAME_FILES);
                        $getFiles = $sourceDBH->query($sql);
                        $success = 0;
                        $error = 0;
                        while ($row = $getFiles->fetch()) {
                            $dbFileStorageValue = $row['file_path'];

                            // append file path
                            if ((int)XFILESHARING_FOLDER_GROUPING > 0) {
                                $actualFolderId = floor((int)$row['file_id'] / 5000);
                                $formattedFolderId = str_pad($actualFolderId, 5, "0", STR_PAD_LEFT);
                                $dbFileStorageValue = $formattedFolderId.'/'.$dbFileStorageValue;
                            }

                            // insert file entry
                            $sql = 'INSERT INTO file (id, originalFilename, shortUrl, extension, userId, totalDownload,
                                              uploadedIP, uploadedDate, statusId, visits, lastAccessed, deleteHash, folderId,
                                              accessPassword)
                            VALUES (:id, :originalFilename, :shortUrl, :extension, :userId, :totalDownload,
                                    :uploadedIP, :uploadedDate, :statusId, :visits, :lastAccessed, :deleteHash, :folderId,
                                    :accessPassword)';
                            $q = $ysDBH->prepare($sql);
                            $count = $q->execute([
                                ':id' => $row['file_id'],
                                ':originalFilename' => $row['file_name'],
                                ':shortUrl' => $row['short_url'],
                                ':extension' => getFileExtension($row['file_name']),
                                ':userId' => ((int)$row['user_id'] == 0 ? null : $row['user_id']),
                                ':totalDownload' => $row['total_downloads'],
                                ':uploadedIP' => long2Ip32bit($row['uploaded_ip']),
                                ':uploadedDate' => $row['created'],
                                ':statusId' => 1,
                                ':visits' => $row['total_downloads'],
                                ':lastAccessed' => $row['last_downloaded'],
                                ':deleteHash' => $row['delete_hash'],
                                ':folderId' => $row['folder_id'],
                                ':accessPassword' => $row['file_password'],
                            ]);

                            // insert file artifact
                            if ($count) {
                                $sql = 'INSERT INTO file_artifact (file_id, local_file_path, file_type, file_hash, file_artifact_type, file_size, created)
                                VALUES (:file_id, :local_file_path, :file_type, :file_hash, "primary", :file_size, :created)';
                                $q = $ysDBH->prepare($sql);
                                $count = $q->execute([
                                    ':file_id' => $row['file_id'],
                                    ':local_file_path' => $dbFileStorageValue,
                                    ':file_type' => guessMimeType($row['file_name']),
                                    ':file_hash' => null, // replace if we know the file content hash (md5)
                                    ':file_size' => $row['file_size'],
                                    ':created' => $row['created'],
                                ]);
                            }

                            // insert file artifact storage entry
                            if ($count) {
                                $artifactId = $ysDBH->lastInsertId();
                                $sql = 'INSERT INTO file_artifact_storage (file_artifact_id, file_server_id, is_primary, created)
                                VALUES (:file_artifact_id, :file_server_id, 1, :created)';
                                $q = $ysDBH->prepare($sql);
                                $count = $q->execute([
                                    ':file_artifact_id' => $artifactId,
                                    ':file_server_id' => $row['server_id'],
                                    ':created' => $row['created'],
                                ]);
                            }

                            if ($count) {
                                $success++;
                            } else {
                                $error++;
                            }
                        }

                        ?>
                        <tr>
                            <td><?php echo SOURCE_TABLE_NAME_FILES; ?>:</td>
                            <td><?php echo $sourceDBStats['totalFiles']; ?></td>
                            <td>file:</td>
                            <td><?php echo $success; ?></td>
                            <td><?php echo $error; ?></td>
                        </tr>
                        <?php
                        updateScreen();
                    }
                    ?>

                    <?php
                    // do users
                    if (isSectionEnabled('users')) {
                        $sql = sprintf('SELECT usr_id as user_id,
                                   usr_login as username,
                                   DECODE(usr_password, \'%s\') as raw_password,
                                   usr_email as email,
                                   usr_adm as user_level,
                                   usr_status as status,
                                   usr_disk_space as disk_space,
                                   usr_premium_expire as premium_expire,
                                   usr_created as created,
                                   usr_lastlogin as lastlogin,
                                   usr_lastip as last_login_ip
                            FROM %s
                            ORDER BY usr_id DESC',
                            XFILESHARING_PASSWORD_HASH, SOURCE_TABLE_NAME_USERS);
                        $getUsers = $sourceDBH->query($sql);
                        $success = 0;
                        $error = 0;
                        while ($row = $getUsers->fetch()) {
                            // insert into Yetishare db
                            $sql = 'INSERT INTO users (id, username, password, level_id, email, lastlogindate, lastloginip, status, storageLimitOverride,
                                               datecreated, createdip, paidExpiryDate, paymentTracker, identifier, apikey, accountLockHash)
                            VALUES (:id, :username, :password, :level_id, :email, :lastlogindate, :lastloginip, :status, :storageLimitOverride,
                                    :datecreated, :createdip, :paidExpiryDate, :paymentTracker, :identifier, md5(paymentTracker), "")';
                            $q = $ysDBH->prepare($sql);
                            $userLevel = 1;
                            if ($row['user_level'] == 1) {
                                $userLevel = 20;
                            } elseif (strtotime($row['premium_expire']) > time()) {
                                $userLevel = 2;
                            }

                            $status = 'active';
                            if ($row['status'] == 'PENDING') {
                                $status = 'pending';
                            } elseif ($row['status'] == 'BANNED') {
                                $status = 'suspended';
                            }

                            $disk_space = 0;
                            if ($row['disk_space'] > 0) {
                                $disk_space = $row['disk_space'] * 1048576;
                            }

                            $count = $q->execute([
                                ':id' => $row['user_id'],
                                ':username' => $row['username'],
                                ':password' => MD5($row['raw_password']),
                                ':level_id' => $userLevel,
                                ':email' => $row['email'],
                                ':lastlogindate' => $row['lastlogin'],
                                ':lastloginip' => long2Ip32bit($row['last_login_ip']),
                                ':status' => $status,
                                ':storageLimitOverride' => $disk_space,
                                ':datecreated' => $row['created'],
                                ':createdip' => long2Ip32bit($row['last_login_ip']),
                                ':paidExpiryDate' => $row['premium_expire'],
                                ':paymentTracker' => md5(microtime().$row['user_id']),
                                ':identifier' => md5(microtime().$row['user_id'].microtime()),
                            ]);

                            if ($q->errorCode() == 0) {
                                $success++;
                            } else {
                                if ($error < 100) {
                                    $errorLocal = $q->errorInfo();
                                    echo 'Skipped Row: '.$errorLocal[2]."<br/>";
                                }
                                if ($error == 100) {
                                    echo "<strong>... [truncated insert errors to first 100]</strong><br/>";
                                }
                                $error++;
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo SOURCE_TABLE_NAME_USERS; ?>:</td>
                            <td><?php echo $sourceDBStats['totalUsers']; ?></td>
                            <td>users:</td>
                            <td><?php echo $success; ?></td>
                            <td><?php echo $error; ?></td>
                        </tr>
                        <?php
                        updateScreen();
                    }
                    ?>

                    <?php
                    // do folders
                    if (isSectionEnabled('folders')) {
                        $sql = sprintf('SELECT fld_id as folder_id, usr_id as user_id, fld_name as folder_name
                            FROM %s',
                            SOURCE_TABLE_NAME_FOLDERS);
                        $getFolders = $sourceDBH->query($sql);
                        $success = 0;
                        $error = 0;
                        while ($row = $getFolders->fetch()) {
                            // insert into Yetishare db
                            $sql = 'INSERT INTO file_folder (id, userId, folderName, isPublic)
                            VALUES (:id, :userId, :folderName, :isPublic)';
                            $q = $ysDBH->prepare($sql);
                            $count = $q->execute([
                                ':id' => $row['folder_id'],
                                ':userId' => $row['user_id'],
                                ':folderName' => $row['folder_name'],
                                ':isPublic' => 0,
                            ]);

                            if ($count) {
                                $success++;
                            } else {
                                $error++;
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo SOURCE_TABLE_NAME_FOLDERS; ?>:</td>
                            <td><?php echo $sourceDBStats['totalFolders']; ?></td>
                            <td>file_folder:</td>
                            <td><?php echo $success; ?></td>
                            <td><?php echo $error; ?></td>
                        </tr>
                        <?php
                        updateScreen();
                    }
                    ?>

                    <?php
                    // do payments
                    if (isSectionEnabled('payments')) {
                        $sql = sprintf('SELECT
                                   id                               as payment_id,
                                   usr_id                           as user_id,
                                   amount                           as amount,
                                   status                           as status,
                                   created                          as created,
                                   CONCAT(pay_type, " ", pay_email) as payment_notes
                            FROM %s
                            WHERE status = \'PAID\'',
                            SOURCE_TABLE_NAME_PAYMENTS);
                        $getPayments = $sourceDBH->query($sql);
                        $success = 0;
                        $error = 0;
                        while ($row = $getPayments->fetch()) {
                            // figure out status
                            $newStatus = 'cancelled';
                            switch ($row['status']) {
                                case 'PENDING':
                                    $newStatus = 'pending';
                                    break;
                                case 'PAID':
                                    $newStatus = 'paid';
                                    break;
                                // 'REJECTED':
                                default:
                                    $newStatus = 'cancelled';
                                    break;
                            }

                            // insert withdraw request
                            $sql = 'INSERT INTO plugin_reward_withdraw_request (id, reward_user_id, requested_date, amount, status, payment_date,
                                                                        payment_notes)
                            VALUES (:id, :reward_user_id, :requested_date, :amount, :status, :payment_date, :payment_notes)';
                            $q = $ysDBH->prepare($sql);
                            $count = $q->execute([
                                ':id' => $row['payment_id'],
                                ':reward_user_id' => $row['user_id'],
                                ':requested_date' => $row['created'],
                                ':amount' => $row['amount'],
                                ':status' => $newStatus,
                                ':payment_date' => $row['created'],
                                ':payment_notes' => $row['payment_notes'],
                            ]);

                            if ($count) {
                                $success++;
                            } else {
                                $error++;
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo SOURCE_TABLE_NAME_PAYMENTS; ?>: (paid)</td>
                            <td><?php echo $sourceDBStats['totalPayments']; ?></td>
                            <td>payment_log:</td>
                            <td><?php echo $success; ?></td>
                            <td><?php echo $error; ?></td>
                        </tr>
                        <?php
                        updateScreen();
                    }
                    ?>
                </table>
                </p>

                <p>
                    Import finished. Note that your admin login to Yetishare will be updated to reflect your
                    old <?php echo SOURCE_APP_NAME; ?> one.
                </p>
                <p style="padding-top: 4px;">
                <form method="POST" action="migrate.php">
                    <input type="submit" value="Restart" name="submit" class="button"/>
                </form>
                </p>
            <?php endif; ?>
        </div>
        <div id="footer">
            Copyright &copy; <?php echo date('Y'); ?> <a href="https://www.yetishare.com"
                                                         target="_blank">Yetishare.com</a>
        </div>
    </div>
    </body>
    </html>

<?php
// local functions
function updateScreen()
{
    flush();
    ob_flush();
}

function getFileExtension($file_name): string
{
    return strtolower(substr(strrchr($file_name, '.'), 1));
}

function long2Ip32bit($ip)
{
    return long2ip((float)$ip);
}

function guessMimeType($filename): string
{
    $mimeTypes = [
        'txt' => 'text/plain',
        'htm' => 'text/html',
        'html' => 'text/html',
        'php' => 'text/html',
        'css' => 'text/css',
        'js' => 'application/javascript',
        'json' => 'application/json',
        'xml' => 'application/xml',
        'swf' => 'application/x-shockwave-flash',
        'flv' => 'video/x-flv',
        // images
        'png' => 'image/png',
        'jpe' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'jpg' => 'image/jpeg',
        'gif' => 'image/gif',
        'bmp' => 'image/bmp',
        'ico' => 'image/vnd.microsoft.icon',
        'tiff' => 'image/tiff',
        'tif' => 'image/tiff',
        'svg' => 'image/svg+xml',
        'svgz' => 'image/svg+xml',
        // archives
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'exe' => 'application/x-msdownload',
        'msi' => 'application/x-msdownload',
        'cab' => 'application/vnd.ms-cab-compressed',
        // audio/video
        'mp3' => 'audio/mpeg',
        'qt' => 'video/quicktime',
        'mov' => 'video/quicktime',
        // adobe
        'pdf' => 'application/pdf',
        'psd' => 'image/vnd.adobe.photoshop',
        'ai' => 'application/postscript',
        'eps' => 'application/postscript',
        'ps' => 'application/postscript',
        // ms office
        'doc' => 'application/msword',
        'rtf' => 'application/rtf',
        'xls' => 'application/vnd.ms-excel',
        'ppt' => 'application/vnd.ms-powerpoint',
        // open office
        'odt' => 'application/vnd.oasis.opendocument.text',
        'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
        'avi' => 'video/avi',
    ];

    $fileParts = explode('.', $filename);
    $ext = strtolower(array_pop($fileParts));
    if (array_key_exists($ext, $mimeTypes)) {
        return $mimeTypes[$ext];
    }

    return 'application/octet-stream';
}

function isSectionEnabled($section): bool
{
    $enabledSection = explode(',', ENABLED_SECTIONS);

    return in_array($section, $enabledSection);
}